<?php defined('BASEPATH') OR exit('No direct script access allowed');

// This can be removed if you use __autoload() in config.php OR use Modular Extensions
require APPPATH.'/libraries/REST_Controller.php';
require APPPATH.'/third_party/Paginator.php';

class Administration extends CI_Controller
{	
	public function __construct(){
		parent::__construct();
		$this->load->library(array('session','encrypt','eventlog','permissions','auth', 'external_notifications'));
		$this->load->model('applicationmodel');
		$this->load->model('applicationrequestmodel');
		$this->load->model('requestmodel');
		$this->load->model('usersmodel');
		$this->load->model('groupsmodel');
		$this->load->helper('url'); 
		date_default_timezone_set(ENVIRONMENT_TIMEZONE);
		
		//check if the user is authorized
		$login_status = $this->session->userdata('is_loggedin');
		if(!isset($login_status) || $login_status !== 'true') {
			//if the user has authenticated via CAC, the controller will load 
			//if not then a 401 Unauthorized error will be shown
			if(!$this->auth->perform_auth()) { show_error('Unauthorized',401); }
		}
		
		//check if user has agreed to dod banner
		$banner = $this->session->userdata('banner_agree');
		if (!isset($banner) || $banner !== 'true') {
			$this->session->set_flashdata('redirect_uri',uri_string());
			redirect('onboarding/banner');
		}
	}
	
	public function index() {
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$show = $this->permissions->set_tab_access_from_permissions($permissions);
		
		if(isset($show['administration']) && $show['administration']) { $this->administration(); }
		else { redirect('onboarding'); }
	}
	
	
	/* -----------------------------*
	 * ADMINISTRATION TAB FUNCTIONS *
	 * -----------------------------*/
	
	/*
	 * Administration is the main landing page for the administration tab. 
	 */
	public function administration() {	
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		
		//use tab access to determine whether to show view at all
		if($data['show']['administration']) { 
			if(isset($permissions['API']['admins']) && $permissions['API']['admins']) {	
				$this->admin_request_list();
			}
			else if(count($this->permissions->get_application_admin_permission($id)) > 0) {
				$this->admin_manage_accounts();
			}
			else { show_error('Forbidden',403); }
		}
		else { show_error('Forbidden',403); }
	}
	
	/*
	 * Admin request list shows all the pending, approved and denied requests.
	 */
	public function admin_request_list() {
		$data['title'] = 'Direct API: Request List';
		$data['active_tab'] = array('administration'=>true);
		
		//grab flash data to display to the user
		if(strlen($this->session->flashdata('message')) > 0) { $data['message'] = $this->session->flashdata('message'); }
		if(strlen($this->session->flashdata('error_message')) > 0) { $data['error_message'] = $this->session->flashdata('error_message'); }
		if(strlen($this->session->flashdata('success_message')) > 0) { $data['success_message'] = $this->session->flashdata('success_message'); }
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		
		
		//pending request list ---------------------------------------------------------
		$pending_pages = new Paginator('pending_');
		$pending_list_size = $this->applicationrequestmodel->get_requests_size(null)->result();
		$pending_list_size = $pending_list_size[0]->count;
		//push back the page number and items per page from the attempt to approve an application to administration/application/requests
		$pending_page_select_from_approve_button = $this->session->flashdata('pending_page_select_from_approve_button');
		$pending_items_per_page_from_approve_button = $this->session->flashdata('pending_items_per_page_from_approve_button');
		if($pending_page_select_from_approve_button) {
			$pending_destination_page = $pending_page_select_from_approve_button;
			//return the pending applications of the page before if the application approved was the only one application on the last page
			if($pending_list_size == ($pending_page_select_from_approve_button - 1) * $pending_items_per_page_from_approve_button) {
				$pending_destination_page = $pending_destination_page - 1;
			}
		}
		if($pending_items_per_page_from_approve_button) {
			$pending_items_per_page = $pending_items_per_page_from_approve_button;
		}
		
		//push back the page number and items per page from the attempt to deny an application to administration/application/requests
		$pending_page_select_from_deny_button = $this->session->flashdata('pending_page_select_from_deny_button');
		$pending_items_per_page_from_deny_button = $this->session->flashdata('pending_items_per_page_from_deny_button');
		if($pending_page_select_from_deny_button) {
			$pending_destination_page = $pending_page_select_from_deny_button;
			//return the pending applications of the page before if the application denied was the only one application on the last page
			if($pending_list_size == ($pending_page_select_from_deny_button - 1) * $pending_items_per_page_from_deny_button) {
				$pending_destination_page = $pending_destination_page - 1;
			}
		}
		if($pending_items_per_page_from_deny_button) {
			$pending_items_per_page = $pending_items_per_page_from_deny_button;
		}
		
		//pending request list pagination setup
		$pending_pages->page_setup($pending_bar_page, $pending_jump_page, $pending_items_per_page, $pending_destination_page);
		$pending_list = $this->applicationrequestmodel->get_requests_by_page_number(null, null, ($pending_destination_page - 1) * $pending_items_per_page + 1, $pending_items_per_page);
		$data['pending_list'] = $pending_list->result();
		$data['pending_items_per_page'] = $pending_items_per_page;
		$data['pending_num_pages'] = ceil($pending_list_size / $pending_items_per_page);
		$data['pending_cur_page'] = $pending_destination_page;
		$pending_pages->items_total = $pending_list_size;
		$pending_pages->current_page = $pending_destination_page;
		$pending_pages->paginate();
		$data['pending_pages'] = $pending_pages;
		
		//approved request list ----------------------------------------------------------
		$approved_pages = new Paginator('approved_');
		$approved_list_size = $this->applicationrequestmodel->get_requests_size('approved')->result();
		$approved_list_size = $approved_list_size[0]->count;
		//push back the page number and items per page on Approved tab
		$approved_page_select_from_approve_button = $this->session->flashdata('approved_page_select_from_approve_button');
		$approved_items_per_page_from_approve_button = $this->session->flashdata('approved_items_per_page_from_approve_button');
		if($approved_page_select_from_approve_button) {
			$approved_destination_page = $approved_page_select_from_approve_button;
			
		}
		if($approved_items_per_page_from_approve_button) {
			$approved_items_per_page = $approved_items_per_page_from_approve_button;
		}
		//push back the page number and items per page on Approved tab
		$approved_page_select_from_deny_button = $this->session->flashdata('approved_page_select_from_deny_button');
		$approved_items_per_page_from_deny_button = $this->session->flashdata('approved_items_per_page_from_deny_button');
		if($approved_page_select_from_deny_button) {
			$approved_destination_page = $approved_page_select_from_deny_button;
			
		}
		if($approved_items_per_page_from_deny_button) {
			$approved_items_per_page = $approved_items_per_page_from_deny_button;
		}
		
		//approved list pagination setup
		$approved_pages->page_setup($approved_bar_page, $approved_jump_page, $approved_items_per_page, $approved_destination_page);
		$approved_list = $this->applicationrequestmodel->get_requests_by_page_number('approved', null, ($approved_destination_page - 1) * $approved_items_per_page + 1, $approved_items_per_page);
		$data['approved_list'] = $approved_list->result();
		$data['approved_items_per_page'] = $approved_items_per_page;
		$data['approved_num_pages'] = ceil($approved_list_size / $approved_items_per_page);
		$data['approved_cur_page'] = $approved_destination_page;
		$approved_pages->items_total = $approved_list_size;
		$approved_pages->current_page = $approved_destination_page;
		$approved_pages->paginate();
		$data['approved_pages'] = $approved_pages;
		
		//denied request list --------------------------------------------------------------
		$denied_pages = new Paginator('denied_');
		$denied_list_size = $this->applicationrequestmodel->get_requests_size('denied')->result();
		$denied_list_size = $denied_list_size[0]->count;
		//push back the page number and items per page on Denied tab
		$denied_page_select_from_approve_button = $this->session->flashdata('denied_page_select_from_approve_button');
		$denied_items_per_page_from_approve_button = $this->session->flashdata('denied_items_per_page_from_approve_button');
		if($denied_page_select_from_approve_button) {
			$denied_destination_page = $denied_page_select_from_approve_button;
			
		}
		if($denied_items_per_page_from_approve_button) {
			$denied_items_per_page = $denied_items_per_page_from_approve_button;
		}
		//push back the page number and items per page on Denied tab
		$denied_page_select_from_deny_button = $this->session->flashdata('denied_page_select_from_deny_button');
		$denied_items_per_page_from_deny_button = $this->session->flashdata('denied_items_per_page_from_deny_button');
		if($denied_page_select_from_deny_button) {
			$denied_destination_page = $denied_page_select_from_deny_button;
			
		}
		if($denied_items_per_page_from_deny_button) {
			$denied_items_per_page = $denied_items_per_page_from_deny_button;
		}
		
		//denied list pagination setup
		$denied_pages->page_setup($denied_bar_page, $denied_jump_page, $denied_items_per_page, $denied_destination_page);
		$denied_list = $this->applicationrequestmodel->get_requests_by_page_number('denied', null, ($denied_destination_page - 1) * $denied_items_per_page + 1, $denied_items_per_page);
		$data['denied_list'] = $denied_list->result();
		$data['denied_items_per_page'] = $denied_items_per_page;
		$data['denied_num_pages'] = ceil($denied_list_size / $denied_items_per_page);
		$data['denied_cur_page'] = $denied_destination_page;
		$denied_pages->items_total = $denied_list_size;
		$denied_pages->current_page = $denied_destination_page;
		$denied_pages->paginate();
		$data['denied_pages'] = $denied_pages;
		
		if(isset($_POST['prefix'])) {
			$data['anchor'] = $_POST['prefix'] . 'link';
		}
		//assign form values
		$form_values = array();
		$form_values['approved_page_select'] = $approved_destination_page;
		$form_values['approved_items_per_page'] = $approved_items_per_page;
		$form_values['pending_page_select'] = $pending_destination_page;
		$form_values['pending_items_per_page'] = $pending_items_per_page;
		$form_values['denied_page_select'] = $denied_destination_page;
		$form_values['denied_items_per_page'] = $denied_items_per_page;
		$data['form_values'] = $form_values;
		
		//use tab access to determine whether to show view at all
		if($data['show']['administration']) { $this->load->view('api/administration/request_list', $data); }
		else { show_error('Forbidden',403); }
	}
	
	/*
	 * Account request list shows all the pending, approved and denied requests for user accounts.
	 */
	public function account_request_list() {
		$this->load->model('accountrequestmodel');
		$data['title'] = 'Direct API: Account Request List';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		
		$approved_list = $this->accountrequestmodel->get_requests('approved');
		$denied_list = $this->accountrequestmodel->get_requests('denied');
		
		//pending request list -----------------------------------------------------------------------------------
		$pending_pages = new Paginator('pending_');
		$pending_pages->page_setup($pending_bar_page, $pending_jump_page, $pending_items_per_page, $pending_destination_page);
		$pending_size = $this->accountrequestmodel->get_requests_size(null)->result();
		$pending_size = $pending_size[0]->count;
		
		//push back the page number and items per page from the attempt to approve an account request to administration/account/requests
		$pending_page_select_from_approve_button = $this->session->flashdata('pending_page_select_from_approve_button');
		$pending_items_per_page_from_approve_button = $this->session->flashdata('pending_items_per_page_from_approve_button');
		if($pending_page_select_from_approve_button) {
			$pending_destination_page = $pending_page_select_from_approve_button;
			//return the pending account requests of the page before if the account approved was the only one account on the last page
			if($pending_size == ($pending_page_select_from_approve_button - 1) * $pending_items_per_page_from_approve_button) {
				$pending_destination_page = $pending_destination_page - 1;
			}
		}
		if($pending_items_per_page_from_approve_button) {
			$pending_items_per_page = $pending_items_per_page_from_approve_button;
		}
		
		//push back the page number and items per page from the attempt to deny an account request to administration/account/requests
		$pending_page_select_from_deny_button = $this->session->flashdata('pending_page_select_from_deny_button');
		$pending_items_per_page_from_deny_button = $this->session->flashdata('pending_items_per_page_from_deny_button');
		if($pending_page_select_from_deny_button) {
			$pending_destination_page = $pending_page_select_from_deny_button;
			//return the pending account request of the page before if the account request denied was the only one on the last page
			if($pending_size == ($pending_page_select_from_deny_button - 1) * $pending_items_per_page_from_deny_button) {
				$pending_destination_page = $pending_destination_page - 1;
			}
		}
		if($pending_items_per_page_from_deny_button) {
			$pending_items_per_page = $pending_items_per_page_from_deny_button;
		}
		
		$pending_list = $this->accountrequestmodel->get_requests_by_page_number(null, ($pending_destination_page - 1) * $pending_items_per_page + 1, $pending_items_per_page);
		$data['pending_list'] = $pending_list->result();
		//pending request list pagination setup
		$data['pending_items_per_page'] = $pending_items_per_page;
		$data['pending_num_pages'] = ceil($pending_size / $pending_items_per_page);
		$data['pending_cur_page'] = $pending_destination_page;
		$pending_pages->items_total = $pending_size;
		$pending_pages->current_page = $pending_destination_page;
		$pending_pages->paginate();
		$data['pending_pages'] = $pending_pages;
		
		//approved request list -----------------------------------------------------------------------------------
		$approved_pages = new Paginator('approved_');
		$approved_size = $this->accountrequestmodel->get_requests_size('approved')->result();
		$approved_size = $approved_size[0]->count;
		//push back the page number and items per page on the Approved tab
		$approved_page_select_from_approve_button = $this->session->flashdata('approved_page_select_from_approve_button');
		$approved_items_per_page_from_approve_button = $this->session->flashdata('approved_items_per_page_from_approve_button');
		if($approved_page_select_from_approve_button) {
			$approved_destination_page = $approved_page_select_from_approve_button;
			
		}
		if($approved_items_per_page_from_approve_button) {
			$approved_items_per_page = $approved_items_per_page_from_approve_button;
		}
		
		//push back the page number and items per page on the Approved tab
		$approved_page_select_from_deny_button = $this->session->flashdata('approved_page_select_from_deny_button');
		$approved_items_per_page_from_deny_button = $this->session->flashdata('approved_items_per_page_from_deny_button');
		if($approved_page_select_from_deny_button) {
			$approved_destination_page = $approved_page_select_from_deny_button;
			
		}
		if($approved_items_per_page_from_deny_button) {
			$approved_items_per_page = $approved_items_per_page_from_deny_button;
		}
		
		//approved request list pagination setup
		$approved_pages->page_setup($approved_bar_page, $approved_jump_page, $approved_items_per_page, $approved_destination_page);
		$approved_list = $this->accountrequestmodel->get_requests_by_page_number('approved', ($approved_destination_page - 1) * $approved_items_per_page + 1, $approved_items_per_page);
		$data['approved_list'] = $approved_list->result();
		$data['approved_items_per_page'] = $approved_items_per_page;
		$data['approved_num_pages'] = ceil($approved_size / $approved_items_per_page);
		$data['approved_cur_page'] = $approved_destination_page;
		$approved_pages->items_total = $approved_size;
		$approved_pages->current_page = $approved_destination_page;
		$approved_pages->paginate();
		$data['approved_pages'] = $approved_pages;
		
		//denied request list -----------------------------------------------------------------------------------
		$denied_pages = new Paginator('denied_');
		$denied_size = $this->accountrequestmodel->get_requests_size('denied')->result();
		$denied_size = $denied_size[0]->count;
		//push back the page number and items per page on the Denied tab
		$denied_page_select_from_approve_button = $this->session->flashdata('denied_page_select_from_approve_button');
		$denied_items_per_page_from_approve_button = $this->session->flashdata('denied_items_per_page_from_approve_button');
		if($denied_page_select_from_approve_button) {
			$denied_destination_page = $denied_page_select_from_approve_button;
			
		}
		if($denied_items_per_page_from_approve_button) {
			$denied_items_per_page = $denied_items_per_page_from_approve_button;
		}
		
		//push back the page number and items per page on the Denied tab
		$denied_page_select_from_deny_button = $this->session->flashdata('denied_page_select_from_deny_button');
		$denied_items_per_page_from_deny_button = $this->session->flashdata('denied_items_per_page_from_deny_button');
		if($denied_page_select_from_deny_button) {
			$denied_destination_page = $denied_page_select_from_deny_button;
			
		}
		if($denied_items_per_page_from_deny_button) {
			$denied_items_per_page = $denied_items_per_page_from_deny_button;
		}
		
		//denied request list pagination setup
		$denied_pages->page_setup($denied_bar_page, $denied_jump_page, $denied_items_per_page, $denied_destination_page);
		$denied_list = $this->accountrequestmodel->get_requests_by_page_number('denied', ($denied_destination_page - 1) * $denied_items_per_page + 1, $denied_items_per_page);
		$data['denied_list'] = $denied_list->result();
		$data['denied_items_per_page'] = $denied_items_per_page;
		$data['denied_num_pages'] = ceil($denied_size / $denied_items_per_page);
		$data['denied_cur_page'] = $denied_destination_page;
		$denied_pages->items_total = $denied_size;
		$denied_pages->current_page = $denied_destination_page;
		$denied_pages->paginate();
		$data['denied_pages'] = $denied_pages;
		
		if(isset($_POST['prefix'])) {
			$data['anchor'] = $_POST['prefix'] . 'link';
		}
		
		//assign form values
		$form_values = array();
		$form_values['approved_page_select'] = $approved_destination_page;
		$form_values['approved_items_per_page'] = $approved_items_per_page;
		$form_values['pending_page_select'] = $pending_destination_page;
		$form_values['pending_items_per_page'] = $pending_items_per_page;
		$form_values['denied_page_select'] = $denied_destination_page;
		$form_values['denied_items_per_page'] = $denied_items_per_page;
		$data['form_values'] = $form_values;
		
		//grab flash data to display to the user
		if(strlen($this->session->flashdata('message')) > 0) { $data['message'] = $this->session->flashdata('message'); }
		if(strlen($this->session->flashdata('error_message')) > 0) { $data['error_message'] = $this->session->flashdata('error_message'); }
		if(strlen($this->session->flashdata('success_message')) > 0) { $data['success_message'] = $this->session->flashdata('success_message'); }
		
		//use tab access to determine whether to show view at all
		if($data['show']['administration']) { $this->load->view('api/administration/account_request_list', $data); }
		else { show_error('Forbidden',403); }
	}
	
	/* This function loads the manage accounts view and provides it with the account data
	 */
	public function admin_manage_accounts() {
		$data['title'] = 'Direct API: Manage Accounts';
		$data['view'] = 'account_list';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$data['create_account_show'] = (isset($permissions['API']['admins']) && $permissions['API']['admins']); //only show create account to API admins
		
		//accounts tab setup -------------------------------------------------------
		$accounts_pages = new Paginator('accounts_');
		$accounts_size = sizeOf($this->usersmodel->get_users());
		
		//push back the page number and items per page from the attempt to disable user to administration/manage/accounts
		$accounts_page_select_from_disable_user_session = $this->session->flashdata('accounts_page_select_from_disable_user_session');
		$accounts_items_per_page_from_disable_user_session = $this->session->flashdata('accounts_items_per_page_from_disable_user_session');
		if($accounts_page_select_from_disable_user_session) {
			$accounts_destination_page = $accounts_page_select_from_disable_user_session;
			//return the accounts of the page before if the account disabled was the only one account on the last page
			if($accounts_size == ($accounts_page_select_from_disable_user_session - 1) * $accounts_items_per_page_from_disable_user_session) {
				$accounts_destination_page = $accounts_destination_page - 1;
			}
		}
		if($accounts_items_per_page_from_disable_user_session) {
			$accounts_items_per_page = $accounts_items_per_page_from_disable_user_session;
		}
		
		//push back the page number and items per page from the attempt to enable user to administration/manage/accounts
		$accounts_page_select_from_enable_user_session = $this->session->flashdata('accounts_page_select_from_enable_user_session');
		$accounts_items_per_page_from_enable_user_session = $this->session->flashdata('accounts_items_per_page_from_enable_user_session');
		if($accounts_page_select_from_enable_user_session) {
			$accounts_destination_page = $accounts_page_select_from_enable_user_session;
			
		}
		if($accounts_items_per_page_from_enable_user_session) {
			$accounts_items_per_page = $accounts_items_per_page_from_enable_user_session;
		}
		
		//accounts tab pagination setup
		$accounts_pages->page_setup($accounts_bar_page, $accounts_jump_page, $accounts_items_per_page, $accounts_destination_page);
		$data['users'] = $this->usersmodel->get_users_by_page_number(($accounts_destination_page - 1) * $accounts_items_per_page + 1, $accounts_items_per_page);
		$data['accounts_items_per_page'] = $accounts_items_per_page;
		$data['accounts_num_pages'] = ceil($accounts_size / $accounts_items_per_page);
		$data['accounts_cur_page'] = $accounts_destination_page;
		$accounts_pages->items_total = $accounts_size;
		$accounts_pages->current_page = $accounts_destination_page;
		$accounts_pages->paginate();
		$data['accounts_pages'] = $accounts_pages;
		
		//deactivated accounts tab setup -------------------------------------------------------
		$deactivated_pages = new Paginator('deactivated_');
		$deactivated_size = sizeOf($this->usersmodel->get_disabled_users());
		
		//push back the page number and items per page on Disabled tab
		$deactivated_page_select_from_enable_user_session = $this->session->flashdata('deactivated_page_select_from_enable_user_session');
		$deactivated_items_per_page_from_enable_user_session = $this->session->flashdata('deactivated_items_per_page_from_enable_user_session');
		if($deactivated_page_select_from_enable_user_session) {
			$deactivated_destination_page = $deactivated_page_select_from_enable_user_session;
			//return the deactivated of the page before if the account enabled was the only one account on the last page
			if($deactivated_size == ($deactivated_page_select_from_enable_user_session - 1) * $deactivated_items_per_page_from_enable_user_session) {
				$deactivated_destination_page = $deactivated_destination_page - 1;
			}
		}
		if($deactivated_items_per_page_from_enable_user_session) {
			$deactivated_items_per_page = $deactivated_items_per_page_from_enable_user_session;
		}
		
		//push back the page number and items per page on Disabled tab
		$deactivated_page_select_from_disable_user_session = $this->session->flashdata('deactivated_page_select_from_disable_user_session');
		$deactivated_items_per_page_from_disable_user_session = $this->session->flashdata('deactivated_items_per_page_from_disable_user_session');
		if($deactivated_page_select_from_disable_user_session) {
			$deactivated_destination_page = $deactivated_page_select_from_disable_user_session;
			
		}
		if($deactivated_items_per_page_from_disable_user_session) {
			$deactivated_items_per_page = $deactivated_items_per_page_from_disable_user_session;
		}
		
		//deactivated accounts tab pagination setup
		$deactivated_pages->page_setup($deactivated_bar_page, $deactivated_jump_page, $deactivated_items_per_page, $deactivated_destination_page);
		$data['disabled_users'] = $this->usersmodel->get_disabled_users_by_page_number(($deactivated_destination_page - 1) * $deactivated_items_per_page + 1, $deactivated_items_per_page);
		$data['deactivated_items_per_page'] = $deactivated_items_per_page;
		$data['deactivated_num_pages'] = ceil($deactivated_size / $deactivated_items_per_page);
		$data['deactivated_cur_page'] = $deactivated_destination_page;
		$deactivated_pages->items_total = $deactivated_size;
		$deactivated_pages->current_page = $deactivated_destination_page;
		$deactivated_pages->paginate();
		$data['deactivated_pages'] = $deactivated_pages;
		
		//create account form data
		$data['first_name'] = $this->session->flashdata('first_name');
		$data['middle_name'] = $this->session->flashdata('middle_name');
		$data['last_name'] = $this->session->flashdata('last_name');
		$data['user_name'] = $this->session->flashdata('user_name');
		$data['user_org_id'] = $this->session->flashdata('user_org_id');
		$data['ext_mail'] = $this->session->flashdata('ext_mail');
		$data['account_title'] = $this->session->flashdata('account_title');
		$data['department'] = $this->session->flashdata('department');
		$data['organization'] = $this->session->flashdata('organization');
		$data['telephone'] = $this->session->flashdata('telephone');
		$data['mobile'] = $this->session->flashdata('mobile');
		$data['location'] = $this->session->flashdata('location');
		
		if(isset($_POST['prefix'])) {
			$data['anchor'] = $_POST['prefix'] . 'link';
		}
		//assign pagination form values
		$form_values = array();
		$form_values['accounts_page_select'] = $accounts_destination_page;
		$form_values['accounts_items_per_page'] = $accounts_items_per_page;
		$form_values['deactivated_page_select'] = $deactivated_destination_page;
		$form_values['deactivated_items_per_page'] = $deactivated_items_per_page;
		$data['form_values'] = $form_values;
		
		//grab flash data to display to the user
		if(strlen($this->session->flashdata('message')) > 0) { $data['message'] = $this->session->flashdata('message'); }
		if(strlen($this->session->flashdata('error_message')) > 0) { $data['error_message'] = $this->session->flashdata('error_message'); }
		if(strlen($this->session->flashdata('success_message')) > 0) { $data['success_message'] = $this->session->flashdata('success_message'); }
		
		$this->load->view('api/administration/manage_accounts', $data);
	}
	
	/* This function loads the edit account view and provides the data for the user
	 * specified by the user id parameter
	 */
	public function admin_edit_account($user_id) {
		$data['title'] = 'Direct API: Manage Accounts';
		$data['view'] = 'account_edit';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$data['user'] = $this->usersmodel->get_user($user_id);
		
		//grab flash data to display to the user
		if(strlen($this->session->flashdata('message')) > 0) { $data['message'] = $this->session->flashdata('message'); }
		if(strlen($this->session->flashdata('error_message')) > 0) { $data['error_message'] = $this->session->flashdata('error_message'); }
		if(strlen($this->session->flashdata('success_message')) > 0) { $data['success_message'] = $this->session->flashdata('success_message'); }
		
		$this->load->view('api/administration/manage_accounts', $data);
	}
	
	/* This function loads the edit account access view and provides the data for the user
	 * specified by the user id parameter
	 */
	public function admin_edit_account_access($user_id) {
		$data['title'] = 'Direct API: Manage Accounts';
		$data['view'] = 'account_edit_access';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$data['user'] = $this->usersmodel->get_user($user_id);
		$data['available_groups'] = $this->usersmodel->get_allowed_groups_for_access($permissions);
		
		$this->load->view('api/administration/manage_accounts', $data);
	}
	
	/* Save account data from account edit form
	 */
	public function account_edit_save() {
		$this->load->library('form_validation');
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$cur_user=$this->usersmodel->get_user_id_from_org_id($id);
		//TO-DO: which groups should have permissions to edit this account data?
		//we may want to check if a user has authorized their account data to be editable by an app and allow it
		//but for now let's restrict it to API admins only
		if($data['show']['administration'] && $data['admin_show']['manage_accounts'] && $permissions['API']['admins']) {
			$first = $this->input->post('first_name', TRUE);
			$middle = $this->input->post('middle_name', TRUE);
			$last = $this->input->post('last_name', TRUE);
			$ext_mail = $this->input->post('ext_mail', TRUE);
			$title = $this->input->post('title', TRUE);
			$department = $this->input->post('department', TRUE);
			$organization = $this->input->post('organization', TRUE);
			$telephone = $this->input->post('telephone', TRUE);
			$mobile = $this->input->post('mobile', TRUE);
			$location = $this->input->post('location', TRUE);
			$user_id = $this->input->post('user_id', TRUE);
			
			$this->form_validation->set_rules('first_name','First Name','required');
			$this->form_validation->set_rules('last_name','Last Name','required');
			$this->form_validation->set_rules('ext_mail','Email','required|valid_email');
			$this->form_validation->set_rules('telephone','Telehone','validate_phone');
			$this->form_validation->set_rules('mobile','Mobile','validate_phone');
			
			if($this->form_validation->run() === TRUE) {
				if(is_numeric($user_id)) {
					$username = $this->usersmodel->get_username_from_id($user_id);
					if($username) {
						$attributes = array(
							'cn' => $first . ' ' . $last,
							'givenName' => $first,
							'initials' => $middle,
							'sn' => $last,
							'title' => $title,
							'departmentNumber' => $department,
							'o' => $organization,
							'telephoneNumber' => $telephone,
							'mobile' => $mobile,
							'physicalDeliveryOfficeName' => $location,
						);
						//set optional attributes / attributes that depend on optional attributes
						if(isset($middle) && strlen(trim($middle)) > 0) { 
							$attributes['initials'] = $middle;
							$attributes['displayName'] = $last.', '.$first.' '.$middle;							
						}
						else { $attributes['displayName'] = $last.', '.$first; }
						$result = $this->usersmodel->update_user($user_id,$ext_mail,$attributes);
						if(!$result || in_array(FALSE,$result)) {
							$this->session->set_flashdata('error_message','Account information failed to saved properly.');
							$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Edit user", time(), 0);
						}
						else { $this->session->set_flashdata('success_message','Account information saved.');  $this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Edit user", time(), 1); }
					}
				}
			}
			else{
				$this->session->set_flashdata('error_message',validation_errors());
				$this->session->set_flashdata('first_name', $first_name);
				$this->session->set_flashdata('middle_name', $middle_name);
				$this->session->set_flashdata('last_name', $last_name);
				$this->session->set_flashdata('user_name', $user_name);
				$this->session->set_flashdata('user_org_id', $user_org_id);
				$this->session->set_flashdata('ext_mail', $mail);
				$this->session->set_flashdata('account_title', $account_title);
				$this->session->set_flashdata('department', $department);
				$this->session->set_flashdata('organization', $organization);
				$this->session->set_flashdata('telephone', $telephone);
				$this->session->set_flashdata('mobile', $mobile);
				$this->session->set_flashdata('location', $location);
			}
			redirect('administration/manage/accounts/edit/'.$user_id);
		}
		else {
			show_error('Forbidden',403);
		}
	}
	
	/* Save account group membership data from account access edit form
	 */
	public function account_edit_access_save() {
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$cur_user=$this->usersmodel->get_user_id_from_org_id($id);
		$groups = $this->input->post('groups',TRUE);
		$action = $this->input->post('action',TRUE); //find out if we are adding or removing
		$user_id = $this->input->post('user_id',TRUE);
		if(!is_numeric($user_id)) { show_404(); } //don't waste time if they are manipulating user id
		
		$username = $this->usersmodel->get_username_from_id($user_id); //easy way to check if user exists
		if($username) {
			foreach($groups as $group) {
				if($action === 'add') { 
					$response = $this->usersmodel->change_group_membership($action,$user_id,$group);
					$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Access added ".$group , time(), $response);
				}
				if($action === 'remove') {
					$response = $this->usersmodel->change_group_membership($action,$user_id,$group);
					$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Access removed ".$group, time(), $response);
				}
			}
		}
		redirect('administration/manage/accounts/edit/access/'.$user_id);
	}
	
	/*
	 * Approve request is called when an admin approves a request and copies the request into the applications table. 
	 */
	public function approve_request()
	{
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$this->load->library('email');
		$permissions = $this->permissions->get_user_permissions($id);
		$user_id=$this->usersmodel->get_user_id_from_org_id($id);
		$request_id = $this->input->post('request_id', TRUE);
		
		//push back the page number and items per page from the attempt to approve an application to administration/application/requests
		$this->session->set_flashdata('pending_page_select_from_approve_button',$this->input->post('pending_page_select_from_approve_button',TRUE));
		$this->session->set_flashdata('pending_items_per_page_from_approve_button',$this->input->post('pending_items_per_page_from_approve_button',TRUE));
		$this->session->set_flashdata('approved_page_select_from_approve_button',$this->input->post('approved_page_select_from_approve_button',TRUE));
		$this->session->set_flashdata('approved_items_per_page_from_approve_button',$this->input->post('approved_items_per_page_from_approve_button',TRUE));
		$this->session->set_flashdata('denied_page_select_from_approve_button',$this->input->post('denied_page_select_from_approve_button',TRUE));
		$this->session->set_flashdata('denied_items_per_page_from_approve_button',$this->input->post('denied_items_per_page_from_approve_button',TRUE));
		
		//use permissions to determine whether to allow function call at all
		if($permissions['API']['admins']) {				
			//make sure the id is specified
			if (strlen(trim($request_id)) > 0){
				
				//get app pocs/requestor emails
				$emails = $this->usersmodel->get_request_emails($request_id);
				
				//grab the request
				$request = $this->applicationrequestmodel->get_request($request_id);
				if($request) {
					//pull the request fields to copy to application
					$request_array = $request->row_array();
					$name = $request_array['name'];
					$url = $request_array['url'];
					$desc = $request_array['description'];
					$poc_name = $request_array['poc_name'];
					$poc_email = $request_array['poc_email'];
					$poc_phone = $request_array['poc_phone'];
					$requestor = $request_array['requestor'];
					
					//create public and private keys
					$public_exist = true;
					$private_exist = true;
					
					//while the public key exists in db generate a new one
					while ($public_exist){
						$public = hash('sha256', openssl_random_pseudo_bytes(32));
						$query = $this->applicationmodel->get_application_from_public($public);
						if ($query !== false && $query->num_rows() === 0){
							$public_exist = false;
						}
					}
					
					//while the private key exists in the db generate a new one
					while ($private_exist){
						$private = hash('sha256', openssl_random_pseudo_bytes(32));
						$query = $this->applicationmodel->get_application_from_private($private);
						if ($query !== false && $query->num_rows() === 0){
							$private_exist = false;
						}
					}
				
					//create the application record
					
					if($this->applicationmodel->create_application($public, $private, $name, $requestor, $url, $desc, $poc_name, $poc_email, $poc_phone, $request_id)) {
						$this->applicationrequestmodel->approve_request($request_id);
						$message = 'Request for application "'.$name.'" has been approved.';
						$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Approve application", time(), 1);
						
						$mail_subject = 'Request for application "'.$name.'" has been approved.';
						$mail_message = "All,\r\n\r\nYour application has been approved for access. Please access the following link to find your public and private keys:\r\n\r\nhttps://".API_ADMINPANEL_DOMAIN."/applications/list\r\n\r\nThank you,\r\n\r\n".EMAIL_SIGNATURE;
						$this->external_notifications->send_message($emails, $mail_subject, $mail_message);
					}
					else { 
						$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Approve application", time(), 0);
						$error_message = 'Request for application "'.$name.'" approval failed to submit.'; 
						$this->session->set_flashdata('error_message',$error_message);
					}
				}
				//note that no request exists for the id
				else {
					$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Approve application", time(), 0);
					$message = 'No request exists for id "'.$request_id.'".';
				}
			}
			//id is a required field
			else {
				$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Approve application", time(), 0);
				$message = 'Request id is a required field.';
			}			
			$this->session->set_flashdata('success_message',$message);
			redirect('administration/application/requests');
		}
		else { 
			$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Approve application", time(), 2);
			show_error('Forbidden',403); 
			}
	}
	
	/* Approve Account Request is called when an admin approves an account request.
     * It uses the Admin API to create a user account using the information provided in the request.
	 */
	public function approve_account_request() {
		$this->load->library('email');
		$this->load->model('accountrequestmodel');
		
		//push back the page number and items per page from the attempt to approve an account to administration/account/requests
		$this->session->set_flashdata('pending_page_select_from_approve_button',$this->input->post('pending_page_select_from_approve_button',TRUE));
		$this->session->set_flashdata('pending_items_per_page_from_approve_button',$this->input->post('pending_items_per_page_from_approve_button',TRUE));
		$this->session->set_flashdata('approved_page_select_from_approve_button',$this->input->post('approved_page_select_from_approve_button',TRUE));
		$this->session->set_flashdata('approved_items_per_page_from_approve_button',$this->input->post('approved_items_per_page_from_approve_button',TRUE));
		$this->session->set_flashdata('denied_page_select_from_approve_button',$this->input->post('denied_page_select_from_approve_button',TRUE));
		$this->session->set_flashdata('denied_items_per_page_from_approve_button',$this->input->post('denied_items_per_page_from_approve_button',TRUE));
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$user_id = $this->usersmodel->get_user_id_from_org_id($id);
		$request_id = $this->input->post('request_id', TRUE);
		//use permissions to determine whether to allow function call at all
		if(isset($permissions['API']['admins']) && $permissions['API']['admins']) {	
			//make sure the id is specified
			if (strlen(trim($request_id)) > 0){
				//grab the request
				$request = $this->accountrequestmodel->get_request($request_id);
				if($request) {
					//pull the request fields to copy to account
					$request_array = $request->row_array();
					
					$fields = array(
						'username' => $this->get_new_username($request_array['first_name'],$request_array['last_name']),
						'first' => $request_array['first_name'],
						'middle' => $request_array['middle_name'],
						'last' => $request_array['last_name'],
						'mail' => $request_array['ext_mail'],
						'title' => $request_array['title'],
						'department' => $request_array['department'],
						'organization' => $request_array['organization'],
						'telephone' => $request_array['telephone'],
						'mobile' => $request_array['mobile'],
						'location' => $request_array['location'],
						'id' => $request_array['user_org_id'],
					);
					
					$emails = $request_array['ext_mail']; //set To line
				
					//mark request approved
					if($this->accountrequestmodel->approve_request($request_id)) {
						$response = $this->create_user($fields);
						$json = json_decode($response['json']);
						if($response['status'] === 200) {
							$mail_subject = 'Request for account has been approved.';
							$mail_message = "Hello,\r\n\r\nYour account has been approved for access.\r\n\r\nThank you,\r\n\r\n".EMAIL_SIGNATURE;
							$this->external_notifications->send_message($emails, $mail_subject, $mail_message);
							
							$target_id = $this->usersmodel->get_user_id_from_org_id($request_array['user_org_id']);
							$this->eventlog->create_event( 3,$target_id ,3 ,$user_id, "Approve user", time(), 1);
							$this->session->set_flashdata('success_message',$json->message);
						}
						else {
							$this->eventlog->create_event( 4,$request_id ,3 ,$user_id, "Approve user", time(), 0);
							$this->accountrequestmodel->unapprove_request($request_id); //set request back to unapproved if create fails
							$this->session->set_flashdata('error_message',$json->message);
						}
					}
					else {
						$this->eventlog->create_event( 4,$request_id ,3 ,$user_id, "Approve user", time(), 0);
						$this->session->set_flashdata('error_message','Failed to approve request.');
					}
				}
				else{
					$message = 'No request exists for id "'.$request_id.'".';
					$this->eventlog->create_event( 4,$request_id ,3 ,$user_id, "Approve user", time(), 0);
				}
			}
			else{
				$this->session->set_flashdata('error_message','Request id is a required field.');
				$this->eventlog->create_event( 4,$request_id ,3 ,$user_id, "Approve user", time(), 0);
			}
			redirect('administration/account/requests');
		}
		else { $this->eventlog->create_event( 4,$request_id ,3 ,$user_id, "Approve user", time(), 2);show_error('Forbidden',403); }
	}
	
	/* This function is called when an admin creates an account manually.
	 * It uses the Admin API to create a user account using the information provided in the form.
	*/
	public function create_account() {
		$this->load->library('form_validation');
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$user_id=$this->usersmodel->get_user_id_from_org_id($id);
			
		//get field values from the form
		$first_name = $this->input->post('first_name', TRUE);
		$middle_name = $this->input->post('middle_name', TRUE);
		$last_name = $this->input->post('last_name', TRUE);
		$user_name = $this->input->post('user_name', TRUE);
		$mail = $this->input->post('ext_mail', TRUE);
		$account_title = $this->input->post('account_title', TRUE);
		$department = $this->input->post('department', TRUE);
		$organization = $this->input->post('organization', TRUE);
		$telephone = $this->input->post('telephone', TRUE);
		$mobile = $this->input->post('mobile', TRUE);
		$location = $this->input->post('location', TRUE);
		$user_org_id = $this->input->post('user_org_id', TRUE);
		
		//use permissions to determine whether to allow function call at all
		if(isset($permissions['API']['admins']) && $permissions['API']['admins']) {
			//set validation for the fields
			$this->form_validation->set_rules('user_name','Username','required|alpha_dash|unique_username|max_length[50]');
			$this->form_validation->set_rules('user_org_id','ID','required|numeric');
			$this->form_validation->set_rules('first_name','First Name','required');
			$this->form_validation->set_rules('last_name','Last Name','required');
			$this->form_validation->set_rules('ext_mail','Email','required|valid_email');
			$this->form_validation->set_rules('telephone','Telehone','validate_phone');
			$this->form_validation->set_rules('mobile','Mobile','validate_phone');
			
			if($this->form_validation->run() === TRUE) {
				$fields = array(
						'username' => $user_name,
						'first' => $first_name,
						'middle' => $middle_name,
						'last' => $last_name,
						'mail' => $mail,
						'title' => $account_title,
						'department' => $department,
						'organization' => $organization,
						'telephone' => $telephone,
						'mobile' => $mobile,
						'location' => $location,
						'id' => $user_org_id,
				);

				//mark account created
				$response = $this->create_user($fields);
				$json = json_decode($response['json']);
				if($response['status'] === 200) {
					$this->load->model('accountrequestmodel');
					$target_id = $this->usersmodel->get_user_id_from_org_id($user_org_id);
					$this->eventlog->create_event(3, $target_id, 3,$user_id, "Create account", time(), 1);
					$this->session->set_flashdata('success_message',$json->message);
					$message = 'Account Name, ' . $user_name . ', for ' . $first_name . ' ' . $last_name . ' has been created successfully.';
					$this->accountrequestmodel->create_request($first_name, $middle_name, $last_name, $mail, $account_title, $department, $organization, $telephone, $mobile, $location , $user_org_id);
					$r_id=$this->accountrequestmodel->get_ids($user_org_id);
					if($r_id){
						$r_id = $r_id->result();
						$this->accountrequestmodel->approve_request($r_id[0]->id);
					}
					else{
						$this->session->set_flashdata('error_message','Failed to find request');
					}
				}
				else {
					$this->eventlog->create_event(3, 0, 3,$user_id, "Create account", time(), 0);
					$this->session->set_flashdata('error_message',$json->message);
				}
				
			}
			else {
				$this->session->set_flashdata('error_message',validation_errors());
				$this->session->set_flashdata('first_name', $first_name);
				$this->session->set_flashdata('middle_name', $middle_name);
				$this->session->set_flashdata('last_name', $last_name);
				$this->session->set_flashdata('user_name', $user_name);
				$this->session->set_flashdata('user_org_id', $user_org_id);
				$this->session->set_flashdata('ext_mail', $mail);
				$this->session->set_flashdata('account_title', $account_title);
				$this->session->set_flashdata('department', $department);
				$this->session->set_flashdata('organization', $organization);
				$this->session->set_flashdata('telephone', $telephone);
				$this->session->set_flashdata('mobile', $mobile);
				$this->session->set_flashdata('location', $location);
			}
		}
		else {
			$this->eventlog->create_event(3, 0, 3 ,$user_id, "Create account", time(), 2);
			show_error(403,'Unauthorized');
		}
		redirect('administration/manage/accounts/#create');
	}
	
	/*
	 * 
	 * Deny request is called when an admin denies an application.
	 * 
	 */
	public function deny_request()
	{
		$this->load->library('email');
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$request_id = $this->input->post('request_id', TRUE);
		$user_id=$this->usersmodel->get_user_id_from_org_id($id);
		
		//push back the page number and items per page from the attempt to deny an application to administration/application/requests
		$this->session->set_flashdata('pending_page_select_from_deny_button',$this->input->post('pending_page_select_from_deny_button',TRUE));
		$this->session->set_flashdata('pending_items_per_page_from_deny_button',$this->input->post('pending_items_per_page_from_deny_button',TRUE));
		$this->session->set_flashdata('approved_page_select_from_deny_button',$this->input->post('approved_page_select_from_deny_button',TRUE));
		$this->session->set_flashdata('approved_items_per_page_from_deny_button',$this->input->post('approved_items_per_page_from_deny_button',TRUE));
		$this->session->set_flashdata('denied_page_select_from_deny_button',$this->input->post('denied_page_select_from_deny_button',TRUE));
		$this->session->set_flashdata('denied_items_per_page_from_deny_button',$this->input->post('denied_items_per_page_from_deny_button',TRUE));
		
		//use permissions to determine whether to allow function call at all
		if($permissions['API']['admins']) {	
			if (strlen(trim($request_id)) > 0){
				
				//get app pocs/requestor emails
				$emails = $this->usersmodel->get_request_emails($request_id);
				
				//grab the request
				$request = $this->applicationrequestmodel->get_request($request_id);
				if($request) {
					$request_array = $request->row_array();
					$name = $request_array['name'];
					$reason = $this->input->post('reason', TRUE);
					
					//process the denial, set a flag in the request table
					if($this->applicationrequestmodel->deny_request($request_id, $reason) !== FALSE) {
						$message = 'Request for application "'.$name.'" has been denied.';
						
						$mail_subject = 'Request for application "'.$name.'" has been denied.';
						$mail_message = "All,\r\n\r\nYour application has been denied access. Please access the following link and click on the denied tab. Then update the request and resubmit.\r\n\r\nhttps://".API_ADMINPANEL_DOMAIN."/applications/list\r\n\r\nThank you,\r\n\r\n".EMAIL_SIGNATURE;
						$this->external_notifications->send_message($emails, $mail_subject, $mail_message);
						
						$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Deny application", time(), 1);
					}
					else {
						$message = 'Failed to deny request for application "'.$name.'".';
						$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Deny application", time(), 0);
					}
				}
				//no request exists for the id
				else{
					$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Deny application", time(), 0);
					$message = 'No request exists for id "'.$request_id.'".';
				}
			}
			//must specify an id
			else{
				$this->eventlog->create_event( 1,$request_id ,3 ,$user_id, "Deny application", time(), 0);
				$message = 'Request id is a required field.';
			}
			$this->session->set_flashdata('message',$message);
			redirect('administration/application/requests');
		}
		else { 
			$this->eventlog->create_event(1, $request_id ,3 ,$user_id, 'Deny app', time(), 2);
			show_error('Forbidden',403); 
		}
	}
	
	/* Deny account request is called when an admin denies a user account.
	 * It set the denied flag for the given request to TRUE in the database.
	 */
	public function deny_account_request() {
		$this->load->library('email');
		$this->load->model('accountrequestmodel');
		
		//push back the page number and items per page from the attempt to deny an account request to administration/account/requests
		$this->session->set_flashdata('pending_page_select_from_deny_button',$this->input->post('pending_page_select_from_deny_button',TRUE));
		$this->session->set_flashdata('pending_items_per_page_from_deny_button',$this->input->post('pending_items_per_page_from_deny_button',TRUE));
		$this->session->set_flashdata('approved_page_select_from_deny_button',$this->input->post('approved_page_select_from_deny_button',TRUE));
		$this->session->set_flashdata('approved_items_per_page_from_deny_button',$this->input->post('approved_items_per_page_from_deny_button',TRUE));
		$this->session->set_flashdata('denied_page_select_from_deny_button',$this->input->post('denied_page_select_from_deny_button',TRUE));
		$this->session->set_flashdata('denied_items_per_page_from_deny_button',$this->input->post('denied_items_per_page_from_deny_button',TRUE));
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$user_id = $this->usersmodel->get_user_id_from_org_id($id);
		$request_id = $this->input->post('request_id', TRUE);
		$request = $this->accountrequestmodel->get_request($request_id)->result();
		//use permissions to determine whether to allow function call at all
		if($permissions['API']['admins']) {	
			$justification = $this->input->post('reason',TRUE);
			
			$emails = $request[0]->ext_mail; //set To line

			//attempt to deny request
			if (strlen(trim($request_id)) > 0){
				if(!$this->accountrequestmodel->deny_request($request_id,$justification)) {
					$this->session->set_flashdata('error_message','Failed to deny account request');
					$this->eventlog->create_event(4, $request_id, 3, $user_id, "Deny user", time(), 0);
				}
				else{
					$name = $request[0]->first_name . ' ' . $request[0]->last_name;
					$message = 'Request for account "'.$name.'" has been denied.';
					
					$mail_subject = 'Request for account has been denied.';
					$mail_message = "Hello,\r\n\r\nYour account request has been denied for the following reason:\r\n".$justification."\r\n\r\nThank you,\r\n\r\n".EMAIL_SIGNATURE;
					$this->external_notifications->send_message($emails, $mail_subject, $mail_message);

					$this->eventlog->create_event(4, $request_id, 3, $user_id, "Deny user", time(), 1);
				}
			}
			else{
				$this->session->set_flashdata('error_message','Request id is a required field');
				$this->eventlog->create_event(4, $request_id, 3, $user_id, "Deny user", time(), 0);
			}
		}
		else{
			$this->eventlog->create_event(4, $request_id, 3, $user_id, "Deny user", time(), 2);
			show_error('Forbidden',403); 
		}
		$this->session->set_flashdata('message',$message);
		redirect('administration/account/requests');
	}
	
	/* This function takes the provided user id from POST input and deactivates the user
	 * associated with it if permissions to do so are present for the current user.
	 */
	public function disable_account() {
		$user_id = $this->input->post('user_id',TRUE);
		
		//get user org id from session to get permissions
		$current_user_org_id = $this->encrypt->decode($this->session->userdata('user_id'));
		$cur_user=$this->usersmodel->get_user_id_from_org_id($current_user_org_id);
		$account_name = $this->usersmodel->get_username_from_id($user_id);
		$permissions = $this->permissions->get_user_permissions($current_user_org_id);
		
		//push back the page number and items per page from the attempt to disable user to administration/manage/accounts
		$this->session->set_flashdata('accounts_page_select_from_disable_user_session',$this->input->post('accounts_page_select_from_disable_user_session',TRUE));
		$this->session->set_flashdata('accounts_items_per_page_from_disable_user_session',$this->input->post('accounts_items_per_page_from_disable_user_session',TRUE));
		$this->session->set_flashdata('deactivated_page_select_from_disable_user_session',$this->input->post('deactivated_page_select_from_disable_user_session',TRUE));
		$this->session->set_flashdata('deactivated_items_per_page_from_disable_user_session',$this->input->post('deactivated_items_per_page_from_disable_user_session',TRUE));
		
		//get org id of the user being acted upon to compare to the current user
		$org_id = $this->usersmodel->get_org_id_from_user_id($user_id);
		if($org_id === $current_user_org_id) { 
			$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Disable account", time(), 0);
			$this->session->set_flashdata('error_message','Cannot disable the logged in account.');
			redirect('administration/manage/accounts');
		}
		else {
			if($permissions['API']['admins']) {	
				if($this->usersmodel->disable_user($user_id)) {
					$this->session->set_flashdata('message','Account '.$account_name.' has been disabled.');
					$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Disable account", time(), 1);
					redirect('administration/manage/accounts');
				}
				else { 
					$this->session->set_flashdata('error_message','Failed to disable account.'); 
					$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Disable account", time(), 0);
					redirect('administration/manage/accounts');
				}
			}
			else { show_error(403,'Forbidden'); $this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Disable account", time(), 2);}
		}
	}
	
	/* This function takes the provided user id from POST input and activates the user
	 * associated with it if permissions to do so are present for the current user.
	 */
	public function enable_account() {
		$user_id = $this->input->post('user_id',TRUE);
		$account_name = $this->usersmodel->get_username_from_id($user_id);
		
		//get user id from session to get permissions
		$current_user_id = $this->encrypt->decode($this->session->userdata('user_id'));
		$cur_user=$this->usersmodel->get_user_id_from_org_id($current_user_id);
		$permissions = $this->permissions->get_user_permissions($current_user_id);
		
		//push back the page number and items per page from the attempt to enable account to administration/manage/accounts
		$this->session->set_flashdata('deactivated_page_select_from_enable_user_session',$this->input->post('deactivated_page_select_from_enable_user_session',TRUE));
		$this->session->set_flashdata('deactivated_items_per_page_from_enable_user_session',$this->input->post('deactivated_items_per_page_from_enable_user_session',TRUE));
		$this->session->set_flashdata('accounts_page_select_from_enable_user_session',$this->input->post('accounts_page_select_from_enable_user_session',TRUE));
		$this->session->set_flashdata('accounts_items_per_page_from_enable_user_session',$this->input->post('accounts_items_per_page_from_enable_user_session',TRUE));
		
		if($permissions['API']['admins']) {	
			if($this->usersmodel->enable_user($user_id)) {
				$this->session->set_flashdata('success_message','Account '.$account_name.' has been enabled.');
				$this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Enable account", time(), 1);
				redirect('administration/manage/accounts#deactivated');
			}
			else { $this->session->set_flashdata('error_message','Failed to enable disabled account.'); $this->eventlog->create_event( 3,$user_id,3 ,$cur_user, "Enable account", time(), 0);}
		}
		else { show_error(403,'Forbidden'); }
	}

	/* This function loads the manage group view and provides it with the account data
	 */
	public function admin_manage_groups() {
	
		$data['title'] = 'Direct API: Manage Groups';
		$data['view'] = 'group_list';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$data['create_group_show'] = (isset($permissions['API']['admins']) && $permissions['API']['admins']); //only show create account to API admins
		
		//accounts tab setup ---------------------------------------------------------
		$accounts_pages = new Paginator('accounts_');
		$accounts_size = sizeOf($this->groupsmodel->get_groups());
		
		//push back the page number and items per page from the attempt to disable group to administration/manage/groups
		$accounts_page_select_from_disable_group_session = $this->session->flashdata('accounts_page_select_from_disable_group_session');
		$accounts_items_per_page_from_disable_group_session = $this->session->flashdata('accounts_items_per_page_from_disable_group_session');
		if($accounts_page_select_from_disable_group_session) {
			$accounts_destination_page = $accounts_page_select_from_disable_group_session;
			//return the groups of the page before if the group disabled was the only one on the last page
			if($accounts_size == ($accounts_page_select_from_disable_group_session - 1) * $accounts_items_per_page_from_disable_group_session) {
				$accounts_destination_page = $accounts_destination_page - 1;
			}
		}
		if($accounts_items_per_page_from_disable_group_session) {
			$accounts_items_per_page = $accounts_items_per_page_from_disable_group_session;
		}
		
		//push back the page number and items per page from the attempt to enable group administration/manage/groups
		$accounts_page_select_from_enable_group_session = $this->session->flashdata('accounts_page_select_from_enable_group_session');
		$accounts_items_per_page_from_enable_group_session = $this->session->flashdata('accounts_items_per_page_from_enable_group_session');
		if($accounts_page_select_from_enable_group_session) {
			$accounts_destination_page = $accounts_page_select_from_enable_group_session;
			//return the groups of the page before if the group enabled was the only one on the last page
			if($accounts_size == ($accounts_page_select_from_enable_group_session - 1) * $accounts_items_per_page_from_enable_group_session) {
				$accounts_destination_page = $accounts_destination_page - 1;
			}
		}
		if($accounts_items_per_page_from_enable_group_session) {
			$accounts_items_per_page = $accounts_items_per_page_from_enable_group_session;
		}
		
		//accounts tab pagination setup
		$accounts_pages->page_setup($accounts_bar_page, $accounts_jump_page, $accounts_items_per_page, $accounts_destination_page);
		$data['groups'] = $this->groupsmodel->get_groups_by_page_number(($accounts_destination_page - 1) * $accounts_items_per_page + 1, $accounts_items_per_page);
		$data['accounts_items_per_page'] = $accounts_items_per_page;
		$data['accounts_num_pages'] = ceil($accounts_size / $accounts_items_per_page);
		$data['accounts_cur_page'] = $accounts_destination_page;
		$accounts_pages->items_total = $accounts_size;
		$accounts_pages->current_page = $accounts_destination_page;
		$accounts_pages->paginate();
		$data['accounts_pages'] = $accounts_pages;
		
		//deactivated accounts tab setup ---------------------------------------------------------
		$deactivated_pages = new Paginator('deactivated_');
		$deactivated_size = sizeOf($this->groupsmodel->get_disabled_groups());
		
		//push back the page number and items per page on Disabled Groups tab
		$deactivated_page_select_from_enable_group_session = $this->session->flashdata('deactivated_page_select_from_enable_group_session');
		$deactivated_items_per_page_from_enable_group_session = $this->session->flashdata('deactivated_items_per_page_from_enable_group_session');
		if($deactivated_page_select_from_enable_group_session) {
			$deactivated_destination_page = $deactivated_page_select_from_enable_group_session;
			//return the deactivated of the page before if the account enabled was the only one account on the last page
			if($deactivated_size == ($deactivated_page_select_from_enable_group_session - 1) * $deactivated_items_per_page_from_enable_group_session) {
				$deactivated_destination_page = $deactivated_destination_page - 1;
			}
		}
		if($deactivated_items_per_page_from_enable_group_session) {
			$deactivated_items_per_page = $deactivated_items_per_page_from_enable_group_session;
		}
		
		//push back the page number and items per page on Disabled Groups tab
		$deactivated_page_select_from_disable_group_session = $this->session->flashdata('deactivated_page_select_from_disable_group_session');
		$deactivated_items_per_page_from_disable_group_session = $this->session->flashdata('deactivated_items_per_page_from_disable_group_session');
		if($deactivated_page_select_from_disable_group_session) {
			$deactivated_destination_page = $deactivated_page_select_from_disable_group_session;
			//return the groups of the page before if the group disabled was the only one on the last page
			if($deactivated_size == ($deactivated_page_select_from_disable_group_session - 1) * $deactivated_items_per_page_from_disable_group_session) {
				$deactivated_destination_page = $deactivated_destination_page - 1;
			}
		}
		if($deactivated_items_per_page_from_disable_group_session) {
			$deactivated_items_per_page = $deactivated_items_per_page_from_disable_group_session;
		}
		
		//deactivated accounts tab pagination setup
		$deactivated_pages->page_setup($deactivated_bar_page, $deactivated_jump_page, $deactivated_items_per_page, $deactivated_destination_page);
		$data['disabled_groups'] = $this->groupsmodel->get_disabled_groups_by_page_number(($deactivated_destination_page - 1) * $deactivated_items_per_page + 1, $deactivated_items_per_page);
		$data['deactivated_items_per_page'] = $deactivated_items_per_page;
		$data['deactivated_num_pages'] = ceil($deactivated_size / $deactivated_items_per_page);
		$data['deactivated_cur_page'] = $deactivated_destination_page;
		$deactivated_pages->items_total = $deactivated_size;
		$deactivated_pages->current_page = $deactivated_destination_page;
		$deactivated_pages->paginate();
		$data['deactivated_pages'] = $deactivated_pages;
		
		//create account form data
		$data['group_name'] = $this->session->flashdata('group_name');
		$data['display_name'] = $this->session->flashdata('display_name');
		$data['description'] = $this->session->flashdata('description');
		
		if(isset($_POST['prefix'])) {
			$data['anchor'] = $_POST['prefix'] . 'link';
		}
		//assign pagination form values
		$form_values = array();
		$form_values['accounts_page_select'] = $accounts_destination_page;
		$form_values['accounts_items_per_page'] = $accounts_items_per_page;
		$form_values['deactivated_page_select'] = $deactivated_destination_page;
		$form_values['deactivated_items_per_page'] = $deactivated_items_per_page;
		$data['form_values'] = $form_values;
		
		//grab flash data to display to the user
		if(strlen($this->session->flashdata('message')) > 0) { $data['message'] = $this->session->flashdata('message'); }
		if(strlen($this->session->flashdata('error_message')) > 0) { $data['error_message'] = $this->session->flashdata('error_message'); }
		if(strlen($this->session->flashdata('success_message')) > 0) { $data['success_message'] = $this->session->flashdata('success_message'); }
		
		$this->load->view('api/administration/manage_groups', $data);
	}
	
	public function create_group() {
		$this->load->library('form_validation');
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$user_id=$this->usersmodel->get_user_id_from_org_id($id);
			
		//get field values from the form
		$group_name = $this->input->post('group_name', TRUE);
		$display_name = $this->input->post('display_name', TRUE);
		$description = $this->input->post('description', TRUE);
		
		//use permissions to determine whether to allow function call at all
		if(isset($permissions['API']['admins']) && $permissions['API']['admins']) {
			//set validation for the fields
			$this->form_validation->set_rules('group_name','Group Name','required|alpha_dash|unique_username|max_length[50]');
			$this->form_validation->set_rules('display_name','Display Name','required|max_length[200]');
			$this->form_validation->set_rules('description','Description','required|max_length[4000]');
			
			if($this->form_validation->run() === TRUE) {
				$fields = array(
						'group_name' => $group_name,
						'display_name' => $display_name,
						'description' => $description,
				);
				
				$response = $this->create_group_account($fields);
				$json = json_decode($response['json']);
				if($response['status'] === 200) {	
					$this->eventlog->create_event(6, 0, 3,$user_id, "Create group: ".$group_name, time(), 1);
					$this->session->set_flashdata('success_message',$json->message);
					$message = 'Group Name, ' . $group_name . ', for ' . $display_name . ' has been created successfully.';
				}
				else {
					$this->eventlog->create_event(6, 0, 3,$user_id, "Create group: ".$group_name, time(), 0);
					$this->session->set_flashdata('error_message',$json->message);
				}
				
			}
			else {
				$this->session->set_flashdata('error_message',validation_errors());
				$this->session->set_flashdata('group_name', $group_name);
				$this->session->set_flashdata('display_name', $display_name);
				$this->session->set_flashdata('description', $description);
			}
		}
		else {
			$this->eventlog->create_event(6, 0, 3 ,$user_id, "Create group: ".$group_name, time(), 2);
			show_error(403,'Unauthorized');
		}
		redirect('administration/manage/groups/#create');
	}
	public function disable_group() {
		$group = $this->input->post('ou',TRUE);
		
		//get user org id from session to get permissions
		$current_user_org_id = $this->encrypt->decode($this->session->userdata('user_id'));
		$cur_user=$this->usersmodel->get_user_id_from_org_id($current_user_org_id);
		$permissions = $this->permissions->get_user_permissions($current_user_org_id);
		
		//push back the page number and items per page from the attempt to disable group to administration/manage/groups
		$this->session->set_flashdata('accounts_page_select_from_disable_group_session',$this->input->post('accounts_page_select_from_disable_group_session',TRUE));
		$this->session->set_flashdata('accounts_items_per_page_from_disable_group_session',$this->input->post('accounts_items_per_page_from_disable_group_session',TRUE));
		$this->session->set_flashdata('deactivated_page_select_from_disable_group_session',$this->input->post('deactivated_page_select_from_disable_group_session',TRUE));
		$this->session->set_flashdata('deactivated_items_per_page_from_disable_group_session',$this->input->post('deactivated_items_per_page_from_disable_group_session',TRUE));
		
		if($permissions['API']['admins']) {	
			if($this->groupsmodel->disable_group($group)) {
				$this->session->set_flashdata('message','Group '.$group.' has been disabled.');
				$this->eventlog->create_event( 6,0,3 ,$cur_user, "Disable group: ".$group, time(), 1);
				redirect('administration/manage/groups');
			}
			else { $this->session->set_flashdata('error_message','Failed to disable group.'); $this->eventlog->create_event( 6,0,3 ,$cur_user, "Disable group: ".$group, time(), 0);}
		}
		else { show_error(403,'Forbidden'); $this->eventlog->create_event( 6,0,3 ,$cur_user, "Disable group: ".$group, time(), 2);}
	
	}
	public function enable_group() {
		$group = $this->input->post('ou',TRUE);
		
		//get user org id from session to get permissions
		$current_user_org_id = $this->encrypt->decode($this->session->userdata('user_id'));
		$cur_user=$this->usersmodel->get_user_id_from_org_id($current_user_org_id);
		$permissions = $this->permissions->get_user_permissions($current_user_org_id);
		
		//push back the page number and items per page from the attempt to enable group to administration/manage/groups
		$this->session->set_flashdata('deactivated_page_select_from_enable_group_session',$this->input->post('deactivated_page_select_from_enable_group_session',TRUE));
		$this->session->set_flashdata('deactivated_items_per_page_from_enable_group_session',$this->input->post('deactivated_items_per_page_from_enable_group_session',TRUE));
		$this->session->set_flashdata('accounts_page_select_from_enable_group_session',$this->input->post('accounts_page_select_from_enable_group_session',TRUE));
		$this->session->set_flashdata('accounts_items_per_page_from_enable_group_session',$this->input->post('accounts_items_per_page_from_enable_group_session',TRUE));
		
		if($permissions['API']['admins']) {	
			if($this->groupsmodel->enable_group($group)) {
				$this->session->set_flashdata('success_message','Group '.$group.' has been enabled.');
				$this->eventlog->create_event( 6,0,3 ,$cur_user, "Enable group: ".$group, time(), 1);
				redirect('administration/manage/groups#deactivated');
			}
			else { $this->session->set_flashdata('error_message','Failed to enable group.'); $this->eventlog->create_event( 6,0,3 ,$cur_user, "Enable group: ".$group, time(), 0);}
		}
		else { show_error(403,'Forbidden'); $this->eventlog->create_event( 6,0,3 ,$cur_user, "Enable group: ".$group, time(), 2);}
	
	}
	/* Save group membership data from group access edit form
	 */
	public function group_edit_access_save() {
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$cur_user=$this->usersmodel->get_user_id_from_org_id($id);
		$groups = $this->input->post('groups',TRUE);
		$action = $this->input->post('action',TRUE); //find out if we are adding or removing
		$group_name = $this->input->post('group_name',TRUE);
		
		$exist = $this->groupsmodel->group_exist($group_name); //easy way to check if group exists
		if($exist) {
			foreach($groups as $group) {
				if($action === 'add') { 
					$response = $this->groupsmodel->change_group_membership($action,$group_name,$group);
					$this->eventlog->create_event( 6,0,3 ,$cur_user, "Access added for ".$group_name.": ".$group , time(), $response);
				}
				if($action === 'remove') {
					$response = $this->groupsmodel->change_group_membership($action,$group_name,$group);
					$this->eventlog->create_event( 6,0,3 ,$cur_user, "Access removed for ".$group_name.": ".$group, time(), $response);
				}
			}
		}
		else{show_404();}
		redirect('administration/manage/groups/edit/access/'.$group_name);
	}
	public function admin_edit_group_access($group_name) {
		$data['title'] = 'Direct API: Manage Groups';
		$data['view'] = 'group_edit_access';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$group = $this->groupsmodel->get_group_from_groupname($group_name);
		if(count($group) === 0){
			$group = $this->groupsmodel->get_group_from_groupname($group_name, false);
		}
		$data['group'] = $group[0];
		$groups = 
		$data['available_groups'] = $this->groupsmodel->get_allowed_groups_for_access($permissions);
		$this->load->view('api/administration/manage_groups', $data);
	}
	public function admin_edit_group($group_name) {
		$data['title'] = 'Direct API: Manage Groups';
		$data['view'] = 'group_edit';
		$data['active_tab'] = array('administration'=>true);
		
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$group = $this->groupsmodel->get_group_from_groupname($group_name);
		if(count($group) === 0){
			$group = $this->groupsmodel->get_group_from_groupname($group_name, false);
		}
		if(isset($group[0])){
			$data['group'] = $group[0];
		}
		else{
			$this->session->set_flashdata('error_message','Group does not exist');
			redirect('administration/manage/groups');
		}
		//grab flash data to display to the user
		if(strlen($this->session->flashdata('message')) > 0) { $data['message'] = $this->session->flashdata('message'); }
		if(strlen($this->session->flashdata('error_message')) > 0) { $data['error_message'] = $this->session->flashdata('error_message'); }
		if(strlen($this->session->flashdata('success_message')) > 0) { $data['success_message'] = $this->session->flashdata('success_message'); }
		
		$this->load->view('api/administration/manage_groups', $data);
	}
	/* Save account data from account edit form
	 */
	public function group_edit_save() {
		$this->load->library('form_validation');
		//get user id from session to get permissions
		$id = $this->encrypt->decode($this->session->userdata('user_id'));
		$permissions = $this->permissions->get_user_permissions($id);
		$data['show'] = $this->permissions->set_tab_access_from_permissions($permissions);
		$data['admin_show'] = $this->permissions->set_admin_access_from_permissions($permissions);
		$cur_user=$this->usersmodel->get_user_id_from_org_id($id);
		//TO-DO: which groups should have permissions to edit this account data?
		//we may want to check if a user has authorized their account data to be editable by an app and allow it
		//but for now let's restrict it to API admins only
		if($data['show']['administration'] && $data['admin_show']['manage_accounts'] && $permissions['API']['admins']) {
			$group_name = $this->input->post('group_name', TRUE);
			$display_name = $this->input->post('display_name', TRUE);
			$description = $this->input->post('description', TRUE);
			$this->form_validation->set_rules('group_name','Group Name','required|alpha_dash|max_length[50]');
			$this->form_validation->set_rules('display_name','Display Name','required|max_length[200]');
			$this->form_validation->set_rules('description','Description','required|max_length[4000]');
			if($this->form_validation->run() === TRUE) {
				if($this->groupsmodel->group_exist($group_name)) {
					$result = $this->groupsmodel->update_group($group_name,$display_name,$description);
					if(!$result) {
						$this->session->set_flashdata('error_message','Group information failed to saved properly.');
						$this->eventlog->create_event( 6,0,3 ,$cur_user, "Edit group: ".$group_name, time(), 0);
					}
					else { $this->session->set_flashdata('success_message','Group information saved.');  $this->eventlog->create_event( 6,0,3 ,$cur_user, "Edit group: ".$group_name, time(), 1); }
				}
			}
			else {
				$this->session->set_flashdata('error_message',validation_errors());
				$this->session->set_flashdata('group_name', $group_name);
				$this->session->set_flashdata('display_name', $display_name);
				$this->session->set_flashdata('description', $description);
			}
			redirect('administration/manage/groups/edit/'.$group_name);
		}
		else {
			show_error('Forbidden',403);
		}
	}
	/* -----------------------------*
	 *  PRIVATE FUNCTIONS           *
	 * -----------------------------*/
	
	private function create_user($fields) {				
		$resource = '/admin/create_user/format/json';
		$url = WEBSERVICE_URL . $resource;
		
		$post = '';
		$content_type = "application/x-www-form-urlencoded";
		foreach($fields as $key=>$value) { $post .= $key.'='.urlencode($value).'&'; } //url-ify the data for the POST
		$post = rtrim($post, '&');
		$headers = array(
			'Authorization: DAAS ' . WEBSERVICE_PUBLIC_KEY . ':'. base64_encode(hash_hmac('sha256',"POST\n" . date('U'). "\n". md5($post) . "\n" . $content_type . "\n" . $resource, WEBSERVICE_PRIVATE_KEY)),
			'Date: ' . date('U'),
			'Content-Type: '. $content_type,
			'Content-Md5: ' . base64_encode(md5($post)),
		);
		
		$ch = curl_init();
		curl_setopt($ch,CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_POST,1);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($ch, CURLOPT_HTTPHEADER,$headers);
		$server_output = curl_exec($ch);
		$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		
		return array('status'=>$http_status,'json'=>$server_output);
	}
	
	private function create_group_account($fields) {				
		$resource = '/admin/create_group/format/json';
		$url = WEBSERVICE_URL . $resource;
		
		$post = '';
		$content_type = "application/x-www-form-urlencoded";
		foreach($fields as $key=>$value) { $post .= $key.'='.urlencode($value).'&'; } //url-ify the data for the POST
		$post = rtrim($post, '&');
		$headers = array(
			'Authorization: DAAS ' . WEBSERVICE_PUBLIC_KEY . ':'. base64_encode(hash_hmac('sha256',"POST\n" . date('U'). "\n". md5($post) . "\n" . $content_type . "\n" . $resource, WEBSERVICE_PRIVATE_KEY)),
			'Date: ' . date('U'),
			'Content-Type: '. $content_type,
			'Content-Md5: ' . base64_encode(md5($post)),
		);
		
		$ch = curl_init();
		curl_setopt($ch,CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_POST,1);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($ch, CURLOPT_HTTPHEADER,$headers);
		$server_output = curl_exec($ch);
		$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		
		return array('status'=>$http_status,'json'=>$server_output);
	}
	
	private function get_new_username($first,$last) {
		$this->load->model('usersmodel');
		//sanitize username input to remove any non-alphanumeric characters
		$first = preg_replace("/[^A-Za-z0-9]/", '', $first);
		$last = preg_replace("/[^A-Za-z0-9]/", '', $last);
		//theoretically we could hit max username lengths for last names, so don't allow more than 100 characters total
		$last = substr($last,0,99);
		
		if((isset($first) && isset($last)) && (strlen($first) > 0 && strlen($last) > 0)) {
			$start = strtolower($first[0] . $last);
			if($this->usersmodel->username_exists($start)) {
				$i = 2;
				while($this->usersmodel->username_exists($start . $i) && $i < 999) {
					$i++;
				}
				if($i <= 999) { return $start . $i; }
				else { return FALSE; } //returns FALSE if more than 999 users have the same combo of first initial / last name
			}
			else { return $start; }
		}
		else { return FALSE; }
	}
	
}

